%% HOBO water level logger subsidence analysis
% -------------------------------------------------------------------------
% This script is part of the supporting information for the paper by van
% Bijsterveldt et al. A preview of the impact of accelerated relative sea
% level rise on densely populated mangrove coasts.
% -------------------------------------------------------------------------
% In this script
% -------------------------------------------------------------------------
% Input:
% - HOBO water level logger files:
%       1. Semarang1819.csv
%       2. Bedonobay1819.csv
%       3. Kanal171819.csv
%       4. Tambakbulasan1819.csv
%       5. Wedung1819.csv
% - coef_1618_0.mat: tidal constituents for Semarang (see 
%   Supporting_script_tidal_analysis.m)
% - UTide Matlab fuctions
%   (Source: https://www.mathworks.com/matlabcentral/fileexchange/46523-
%   utide-unified-tidal-analysis-and-prediction-functions)
% Analysis:
% - Remove datapoints when logger was emerged (check ~= TRUE) or water
%   depth smaller than 0.05 m
% - Calculate predicted water levels based on astronomic tide
% - Estimate sensor height of each HOBO based on submergence time during
%   first 90 days of deployment by comparing with measured submergence time
%   to predicted submergence time based on astronomic tide
% - Estimate subsidence of each logger by comparing changes in average
%   water level and submergence time over same period between 2018 and
%   2019. Four periods are compared:
%       A: 21 April - 21 June
%       B: 20 June - 20 August
%       C: 19 August - 19 October
%       D: 21 April - 7 May
%   All sensors were compared over period D. Only the sensors in Semarang, 
%   Kanal, Tambakbulasan and Wedung had long enough time series to allow 
%   for the comparison over periods A, B and C.
% Output:
% - Figure:     HOBO water level data of Semarang, adjusted for the 
%               estimated sensor height and compared to the predicted tidal
%               water levels
% - .csv files: comparison values between 2018 and 2019 for each logger (5)
%               and each comparison period (A, B, C and D), containing the
%               following data:
%                   - Days comparison period [UTC] (2018)
%                   - Daily mean water level, measured [m] (2018)
%                   - Daily mean water level, predicted [m] (2018)
%                   - Daily submergence time, measured [h] (2018)
%                   - Daily submergence time, predicted [h] (2018)
%                   - Days comparison period [UTC] (2019)
%                   - Daily mean water level, measured [m] (2019)
%                   - Daily mean water level, predicted [m] (2019)
%                   - Daily submergence time, measured [h] (2019)
%                   - Daily submergence time, predicted [h] (2019)
% -------------------------------------------------------------------------
% Silke Tas, 22 January 2023 (silke@bu.edu)
% -------------------------------------------------------------------------
% Clear workspace
clear variables
% Define paths and filenames (ADAPT TO OWN DIRECTORIES)
PATH = ['/Users/silke/Library/CloudStorage/OneDrive-BostonUniversity/', ...
        'Research/Matlab/TU/Water level loggers/'];
addpath(genpath('Tools/UTide'));
filenames = {'Semarang1819';
             'Bedonobay1819';
             'Kanal171819';
             'Tambakbulasan1819';
             'Wedung1819'};
coefname = ['/Users/silke/Library/CloudStorage/OneDrive-', ...
            'BostonUniversity/Research/Matlab/TU/Tide/coef_1618_0.mat'];
figname = 'TU/Plots/HOBO_Semarang';
% Options
nd = 90;            % Number of days for submergence time analysis
%% Read HOBO water level logger data
% Loop through loggers
for ii = 1:length(filenames)
    fprintf('Reading HOBO file %s...\n', char(filenames{ii}(1:end-4)));
    % Read file
    D = readtable(char(strcat(PATH, filenames(ii), '.csv')), ...
                  'VariableNamingRule', 'preserve');
    % Define vectors
    t = table2array(D(:, 1)) - hours(7);        %#ok<NASGU> 
    p = table2array(D(:, 2));                   %#ok<NASGU> 
    h = table2array(D(:, 3));                   %#ok<NASGU> 
    c = table2array(D(:, 4));                   %#ok<NASGU> 
    % Save to structure
    eval(sprintf('H.L_%02u.t = t;', ii));       % Time [UTC]
    eval(sprintf('H.L_%02u.p = p;', ii));       % Pressure [kPa]
    eval(sprintf('H.L_%02u.h = h;', ii));       % Water depth [m]
    eval(sprintf('H.L_%02u.c = c;', ii));       % Submerged [true/false]
    eval(sprintf('H.L_%02u.n = char(filenames(ii));', ii));
                                                % Logger ID
    % Clear memory
    clear D t p h c
end
% Clear memory
clear ii
%% Remove data when water depth < 0.05 m
fprintf('Removing data when water depth < 0.05 m...\n');
% Loop through loggers
for ii = 1:length(filenames)
    % Remove data when logger not submerged (check ~= TRUE)
    eval(sprintf('H.L_%02u.p(H.L_%02u.c ~= "TRUE") = NaN;', ii, ii));
    eval(sprintf('H.L_%02u.h(H.L_%02u.c ~= "TRUE") = NaN;', ii, ii));
    % Remove data when water depth < 0.05 m
    eval(sprintf('H.L_%02u.p(H.L_%02u.h < 0.05) = NaN;', ii, ii));
    eval(sprintf('H.L_%02u.h(H.L_%02u.h < 0.05) = NaN;', ii, ii));
    % Remove data when water depth > 2 m
    eval(sprintf('H.L_%02u.p(H.L_%02u.h > 2) = NaN;', ii, ii));
    eval(sprintf('H.L_%02u.h(H.L_%02u.h > 2) = NaN;', ii, ii));
    % Remove NaNs at start and end of deployment
    eval(sprintf('j0 = find(~isnan(H.L_%02u.p), 1, "first");', ii));
    eval(sprintf('j1 = find(~isnan(H.L_%02u.p), 1, "last");', ii));
    eval(sprintf('H.L_%02u.t = H.L_%02u.t(j0:j1);', ii, ii));
    eval(sprintf('H.L_%02u.p = H.L_%02u.p(j0:j1);', ii, ii));
    eval(sprintf('H.L_%02u.h = H.L_%02u.h(j0:j1);', ii, ii));
    eval(sprintf('H.L_%02u.c = H.L_%02u.c(j0:j1);', ii, ii));
    % Clear memory
    clear j0 j1
end
% Clear memory
clear ii
%% Calculate predicted water levels based on astronomic tide
fprintf('Calculate predicted water levels based on astronomic tide...\n');
% Load tidal constituents
load(coefname);
% Loop through loggers
for ii = 1:length(filenames)
    % Use UTide to calculate astronomic tide during logger time series
    eval(sprintf(['H.L_%02u.tide = ut_reconstr(datenum(H.L_%02u.t), ', ...
                  'coef_1618_0);'], ii, ii));
end
% Clear memory
clear coefname coef_1618_0 ii
%% Estimate sensor height above MSL based on submergence time
% Loop through loggers
for ii = 1:length(filenames)
    % Get vectors for current logger
    eval(sprintf('t = datenum(H.L_%02u.t);', ii));
    eval(sprintf('h = H.L_%02u.h;', ii));
    eval(sprintf('tide = H.L_%02u.tide;', ii));
    % Select first 90 days of deployment
    td = (ceil(t(1)) : 1 : ceil(t(1)) + nd - 1)';
    % Determine submergence time per day
    dz       = 0.2;                     % First guess sensor elevation [m]
    t_sub    = zeros(length(nd), 1);
    tide_sub = zeros(length(nd), 1);
    for jj = 1:nd
        t_sub(jj)    = sum(~isnan(h(t >= td(jj) & t < td(jj) + 1)));
        tide_sub(jj) = sum(tide(t >= td(jj) & t < td(jj) + 1) > dz + 0.05);
    end
    % Estimate sensor height to obtain same submergence time based on
    % astronomic water levels
    while abs(mean(t_sub) - mean(tide_sub)) > 0.1
        % New guess sensor elevation
        dz = dz + 0.0001*sign(mean(tide_sub) - mean(t_sub));
        % Re-calculate submergence time based on new guess sensor
        for jj = 1:nd
            tide_sub(jj) = sum(tide(t >= td(jj) & t < td(jj) + 1) > dz + 0.05);
        end
    end
    fprintf('%s: Estimated sensor height above MSL: %.2f m\n', ...
            char(filenames{ii}(1:end-4)), dz);
    % Save sensor height
    eval(sprintf('H.L_%02u.dz = dz;', ii));
    % Clear memory
    clear t h tide td dz t_sub tide_sub jj
end
% Clear memory
clear ii nd
%% Figure: Measured and predicted water level Semarang, est. sensor height
fprintf('Plotting measured and predicted water level Semarang...\n');
% Create new figure
figure;
set(gcf, 'PaperUnits', 'centimeters', ...
         'PaperOrientation', 'portrait', ...
         'PaperType','A4', ...
         'PaperPosition', [1 1 20 14]);
% Get vectors for current logger
eval(sprintf('t = datenum(H.L_%02u.t);', 1));
eval(sprintf('h = H.L_%02u.h;', 1));
eval(sprintf('tide = H.L_%02u.tide;', 1));
eval(sprintf('dz = H.L_%02u.dz;', 1));
% Plot data
plot(t, tide);
hold on
plot(t, h + dz);
rectangle('Position', [t(1) dz+0.05 90 0.35], ...
          'EdgeColor', 'k', ...
          'LineWidth', 1);
% Plot lay-out
xlim([floor(t(1)) ceil(t(end))]);
ylim([-0.6 0.7]);
ax = gca;
ax.XTick = [datenum(2018, 05:12, 01) datenum(2019, 01:10, 01)]; %#ok<DATNM> 
ax.XTickLabel = {'', '', datestr(datenum(2018, 07, 01), 'mmm/yy'), ...
                 '', '', datestr(datenum(2018, 10, 01), 'mmm/yy'), ...
                 '', '', datestr(datenum(2019, 01, 01), 'mmm/yy'), ...
                 '', '', datestr(datenum(2019, 04, 01), 'mmm/yy'), ...
                 '', '', datestr(datenum(2019, 07, 01), 'mmm/yy'), ...
                 '', '', datestr(datenum(2019, 10, 01), 'mmm/yy')}; ...
                                                        %#ok<DATNM,DATST> 
ax.XTickLabelRotation = 0;
ylabel('Water level [m]');
legend('Tidal curve', 'Water level logger', 'Location', 'southeast');
title('Semarang 2018-2019');
% Save figure
print(figname, '-depsc');
print(figname, '-dpng');
% Clear memory
clear figname ax
%% Compare mean water level and submergence time for same period 2018-2019
fprintf('Comparing recorded and predicted water levels 2018-2019...\n');
% Define time period A for comparison: 21 April - 21 June
tr(1, 1, 1) = datenum(2018, 04, 21);                        %#ok<DATNM> 
tr(1, 1, 2) = datenum(2018, 06, 21);                        %#ok<DATNM> 
tr(1, 2, 1) = datenum(2019, 04, 21);                        %#ok<DATNM> 
tr(1, 2, 2) = datenum(2019, 06, 21);                        %#ok<DATNM> 
% Define time period B for comparison: 20 June - 20 August
tr(2, 1, 1) = datenum(2018, 06, 20);                        %#ok<DATNM> 
tr(2, 1, 2) = datenum(2018, 08, 20);                        %#ok<DATNM> 
tr(2, 2, 1) = datenum(2019, 06, 20);                        %#ok<DATNM> 
tr(2, 2, 2) = datenum(2019, 08, 20);                        %#ok<DATNM> 
% Define time period C for comparison: 19 August - 19 October
tr(3, 1, 1) = datenum(2018, 08, 19);                        %#ok<DATNM> 
tr(3, 1, 2) = datenum(2018, 10, 19);                        %#ok<DATNM> 
tr(3, 2, 1) = datenum(2019, 08, 19);                        %#ok<DATNM> 
tr(3, 2, 2) = datenum(2019, 10, 19);                        %#ok<DATNM> 
% Define time period D for comparison: 21 April - 7 May
tr(4, 1, 1) = datenum(2018, 04, 21);                        %#ok<DATNM> 
tr(4, 1, 2) = datenum(2018, 05, 07);                        %#ok<DATNM> 
tr(4, 2, 1) = datenum(2019, 04, 21);                        %#ok<DATNM> 
tr(4, 2, 2) = datenum(2019, 05, 07);                        %#ok<DATNM> 
% Define empty matrix to store comparison values
    % Dim 1: HOBO logger
    % Dim 2: Mean water level (1) or submergence time (2)
    % Dim 3: Measured (1) or predicted (2)
    % Dim 4: 2018 (1) or (2019)
    % Dim 5: period A (1), period B (2) or period C (3)
    % Dim 6: daily values
comp = zeros(length(filenames), 2, 2, 2, size(tr, 1), 61);
% Loop through loggers and comparison periods
for ii = 1:length(filenames)
    % Get vectors for current logger
    eval(sprintf('t = datenum(H.L_%02u.t);', ii));
    eval(sprintf('h = H.L_%02u.h;', ii));
    eval(sprintf('tide = H.L_%02u.tide;', ii));
    eval(sprintf('dz = H.L_%02u.dz;', ii));
    if ii ~= 2 % Exclude second logger (too short measurement period)
        for cc = 1:size(tr, 1)
            % Select data for current comparsion period
            t_18    = t(t > tr(cc, 1, 1) & t < tr(cc, 1, 2));
            t_19    = t(t > tr(cc, 2, 1) & t < tr(cc, 2, 2));
            h_18    = h(t > tr(cc, 1, 1) & t < tr(cc, 1, 2));
            h_19    = h(t > tr(cc, 2, 1) & t < tr(cc, 2, 2));
            tide_18 = tide(t > tr(cc, 1, 1) & t < tr(cc, 1, 2));
            tide_19 = tide(t > tr(cc, 2, 1) & t < tr(cc, 2, 2));
            % Reduce tidal curve to water levels above sensor height
            tide_18(tide_18 <= dz + 0.05) = NaN;
            tide_19(tide_19 <= dz + 0.05) = NaN;
            % Loop through days
            for dd = 1:tr(1, 1, 2) - tr(1, 1, 1)
                % Mean HOBO water level per day (2018)
                comp(ii, 1, 1, 1, cc, dd) = ...
                    mean(h_18(t_18 >= t_18(1) + dd - 1 & ...
                              t_18 <  t_18(1) + dd), ...
                         'omitnan');
                % Mean HOBO water level per day (2018)
                comp(ii, 1, 1, 2, cc, dd) = ...
                    mean(h_19(t_19 >= t_19(1) + dd - 1 & ...
                              t_19 <  t_19(1) + dd), ...
                         'omitnan');
                % Mean tide water level per day (2018)
                comp(ii, 1, 2, 1, cc, dd) = ...
                    mean(tide_18(t_18 >= t_18(1) + dd - 1 & ...
                                 t_18 <  t_18(1) + dd), ...
                         'omitnan');
                % Mean tide water level per day (2019)
                comp(ii, 1, 2, 2, cc, dd) = ...
                    mean(tide_19(t_19 >= t_19(1) + dd - 1 & ...
                                 t_19 <  t_19(1) + dd), ...
                         'omitnan');
                % Submergence time per day based on HOBO water levels ('18)
                comp(ii, 2, 1, 1, cc, dd) = ...
                    sum(~isnan(h_18(t_18 >= t_18(1) + dd - 1 & ...
                                    t_18 <  t_18(1) + dd)));
                % Submergence time per day based on HOBO water levels ('19)
                comp(ii, 2, 1, 2, cc, dd) = ...
                    sum(~isnan(h_19(t_19 >= t_19(1) + dd - 1 & ...
                                    t_19 <  t_19(1) + dd)));
                % Submergence time per day based on tide water levels ('18)
                comp(ii, 2, 2, 1, cc, dd) = ...
                    sum(~isnan(tide_18(t_18 >= t_18(1) + dd - 1 & ...
                                       t_18 <  t_18(1) + dd)));
                % Submergence time per day based on tide water levels ('19)
                comp(ii, 2, 2, 2, cc, dd) = ...
                    sum(~isnan(tide_19(t_19 >= t_19(1) + dd - 1 & ...
                                       t_19 <  t_19(1) + dd)));
            end
            % Clear memory
            clear t_18 t_19 h_18 h_19 tide_18 tide_19
        end
    else
        for cc = 4
            % Select data for current comparsion period
            t_18    = t(t > tr(cc, 1, 1) & t < tr(cc, 1, 2));
            t_19    = t(t > tr(cc, 2, 1) & t < tr(cc, 2, 2));
            h_18    = h(t > tr(cc, 1, 1) & t < tr(cc, 1, 2));
            h_19    = h(t > tr(cc, 2, 1) & t < tr(cc, 2, 2));
            tide_18 = tide(t > tr(cc, 1, 1) & t < tr(cc, 1, 2));
            tide_19 = tide(t > tr(cc, 2, 1) & t < tr(cc, 2, 2));
            % Reduce tidal curve to water levels above sensor height
            tide_18(tide_18 <= dz + 0.05) = NaN;
            tide_19(tide_19 <= dz + 0.05) = NaN;
            % Loop through days
            for dd = 1:tr(1, 1, 2) - tr(1, 1, 1)
                % Mean HOBO water level per day (2018)
                comp(ii, 1, 1, 1, cc, dd) = ...
                    mean(h_18(t_18 >= t_18(1) + dd - 1 & ...
                              t_18 <  t_18(1) + dd), ...
                         'omitnan');
                % Mean HOBO water level per day (2018)
                comp(ii, 1, 1, 2, cc, dd) = ...
                    mean(h_19(t_19 >= t_19(1) + dd - 1 & ...
                              t_19 <  t_19(1) + dd), ...
                         'omitnan');
                % Mean tide water level per day (2018)
                comp(ii, 1, 2, 1, cc, dd) = ...
                    mean(tide_18(t_18 >= t_18(1) + dd - 1 & ...
                                 t_18 <  t_18(1) + dd), ...
                         'omitnan');
                % Mean tide water level per day (2019)
                comp(ii, 1, 2, 2, cc, dd) = ...
                    mean(tide_19(t_19 >= t_19(1) + dd - 1 & ...
                                 t_19 <  t_19(1) + dd), ...
                         'omitnan');
                % Submergence time per day based on HOBO water levels ('18)
                comp(ii, 2, 1, 1, cc, dd) = ...
                    sum(~isnan(h_18(t_18 >= t_18(1) + dd - 1 & ...
                                    t_18 <  t_18(1) + dd)));
                % Submergence time per day based on HOBO water levels ('19)
                comp(ii, 2, 1, 2, cc, dd) = ...
                    sum(~isnan(h_19(t_19 >= t_19(1) + dd - 1 & ...
                                    t_19 <  t_19(1) + dd)));
                % Submergence time per day based on tide water levels ('18)
                comp(ii, 2, 2, 1, cc, dd) = ...
                    sum(~isnan(tide_18(t_18 >= t_18(1) + dd - 1 & ...
                                       t_18 <  t_18(1) + dd)));
                % Submergence time per day based on tide water levels ('19)
                comp(ii, 2, 2, 2, cc, dd) = ...
                    sum(~isnan(tide_19(t_19 >= t_19(1) + dd - 1 & ...
                                       t_19 <  t_19(1) + dd)));
            end
            % Clear memory
            clear t_18 t_19 h_18 h_19 tide_18 tide_19
        end
    end
    % Clear memory
    clear t h tide dz
end
% Clear memory
clear ii cc dd
%% Export comparison values to .csv
fprintf('Exporting comparison values to .csv...\n');
% Letters corresponding to comparison periods
trletters = 'abcd';
% Loop through loggers
for ii = 1:length(filenames)
    % Exclude Bedono Bay from 60-day comparison periods
    if ii ~= 2
        for cc = 1:3
            % Create table with comparison values
            D = table(datetime(datestr((tr(cc, 1, 1):1:...
                                        tr(cc, 1, 2) - 1)')), ...
                      squeeze(comp(ii, 1, 1, 1, cc, :)), ...
                      squeeze(comp(ii, 1, 2, 1, cc, :)), ...
                      squeeze(comp(ii, 2, 1, 1, cc, :)/4), ...
                      squeeze(comp(ii, 2, 2, 1, cc, :)/4), ...
                      datetime(datestr((tr(cc, 2, 1):1:...
                                        tr(cc, 2, 2) - 1)')), ...
                      squeeze(comp(ii, 1, 1, 2, cc, :)), ...
                      squeeze(comp(ii, 1, 2, 2, cc, :)), ...
                      squeeze(comp(ii, 2, 1, 2, cc, :)/4), ...
                      squeeze(comp(ii, 2, 2, 2, cc, :)/4), ...
                      'VariableNames', ...
                      ["Time_UTC_2018", ...
                       "Mean_logger_m_2018", ...
                       "Mean_tide_m_2018", ...
                       "Submergence_time_logger_h_2018", ...
                       "Submergence_time_tide_h_2018", ...
                       "Time_UTC_2019", ...
                       "Mean_logger_m_2019", ...
                       "Mean_tide_m_2019", ...
                       "Submergence_time_logger_h_2019", ...
                       "Submergence_time_tide_h_2019"]); %#ok<DATST> 
            % Write table to .csv file
            eval(sprintf(['writetable(D, char(strcat(PATH, ', ...
                          '"Comp_%s_60%s.csv")));'], ...
                         filenames{ii}(1:end-4), trletters(cc)));
            % Clear memory
            clear D
        end
    end
    % 15-day comparison period for all loggers
        for cc = 4
            % Create table with comparison values
            D = table(datetime(datestr((tr(cc, 1, 1):1:...
                                        tr(cc, 1, 2) - 1)')), ...
                      squeeze(comp(ii, 1, 1, 1, cc, 1:16)), ...
                      squeeze(comp(ii, 1, 2, 1, cc, 1:16)), ...
                      squeeze(comp(ii, 2, 1, 1, cc, 1:16)/4), ...
                      squeeze(comp(ii, 2, 2, 1, cc, 1:16)/4), ...
                      datetime(datestr((tr(cc, 2, 1):1:...
                                        tr(cc, 2, 2) - 1)')), ...
                      squeeze(comp(ii, 1, 1, 2, cc, 1:16)), ...
                      squeeze(comp(ii, 1, 2, 2, cc, 1:16)), ...
                      squeeze(comp(ii, 2, 1, 2, cc, 1:16)/4), ...
                      squeeze(comp(ii, 2, 2, 2, cc, 1:16)/4), ...
                      'VariableNames', ...
                      ["Time_UTC_2018", ...
                       "Mean_logger_m_2018", ...
                       "Mean_tide_m_2018", ...
                       "Submergence_time_logger_h_2018", ...
                       "Submergence_time_tide_h_2018", ...
                       "Time_UTC_2019", ...
                       "Mean_logger_m_2019", ...
                       "Mean_tide_m_2019", ...
                       "Submergence_time_logger_h_2019", ...
                       "Submergence_time_tide_h_2019"]); %#ok<DATST> 
            % Write table to .csv file
            eval(sprintf(['writetable(D, char(strcat(PATH, ', ...
                          '"Comp_%s_15%s.csv")));'], ...
                         filenames{ii}(1:end-4), trletters(cc)));
            % Clear memory
            clear D
        end
end
% Clear memory
clear ii cc trletters tr PATH filenames H comp